//--------------------------------------------------------------------------
// Copyright (C) 2019-2021 Cisco and/or its affiliates. All rights reserved.
//
// This file may contain proprietary rules that were created, tested and
// certified by Sourcefire, Inc. (the "VRT Certified Rules") as well as
// rules that were created by Sourcefire and other third parties and
// distributed under the GNU General Public License (the "GPL Rules").
// The VRT Certified Rules contained in this file are the property of
// Sourcefire, Inc. Copyright 2005 Sourcefire, Inc. All Rights Reserved.
// The GPL Rules created by Sourcefire, Inc. are the property of
// Sourcefire, Inc. Copyright 2002-2005 Sourcefire, Inc. All Rights
// Reserved. All other GPL Rules are owned and copyrighted by their
// respective owners (please see www.snort.org/contributors for a list
// of owners and their respective copyrights). In order to determine what
// rules are VRT Certified Rules or GPL Rules, please refer to the VRT
// Certified Rules License Agreement.
//--------------------------------------------------------------------------
// file-flash_cve-2007-0071-swf-definesceneandframelabeldata-rce.cc author Brandon Stultz <brastult@cisco.com>

#include "main/snort_types.h"
#include "framework/so_rule.h"
#include "framework/cursor.h"
#include "protocols/packet.h"
#include "util_read.h"

#include <cmath>

//#define DEBUG
#ifdef DEBUG
#define DEBUG_SO(code) code
#else
#define DEBUG_SO(code)
#endif

using namespace snort;

static const char* rule_17647 = R"[Snort_SO_Rule](
alert file (
	msg:"FILE-FLASH Adobe Flash Player DefineSceneAndFrameLabelData memory corruption attempt";
	soid:17647;
	file_data;
	content:"FWS",depth 3;
	so:eval;
	metadata:policy max-detect-ips drop, policy security-ips drop;
	reference:bugtraq,28695;
	reference:cve,2007-0071;
	reference:url,www.adobe.com/support/security/bulletins/apsb08-11.html;
	classtype:attempted-user;
	gid:3; sid:17647; rev:8;
)
)[Snort_SO_Rule]";

static const unsigned rule_17647_len = 0;

static IpsOption::EvalStatus eval(void*, Cursor& c, Packet* p)
{
   const uint8_t *cursor_normal = c.start(),
                 *end_of_buffer = c.endo();

   uint8_t nbits;
   uint16_t rect_size, tag, tag_type;
   uint32_t tag_len;

   // skip:
   //  version     (1 byte)
   //  file length (4 bytes)
   cursor_normal += 5;

   // check if we can read:
   //  rect.nbits (1 byte)
   if(cursor_normal + 1 > end_of_buffer)
      return IpsOption::NO_MATCH;

   // read nbits
   nbits = (*cursor_normal & 0xF8) >> 3;

   // calculate rect_size
   rect_size = (uint16_t)ceil((double)(5 + (nbits * 4))/8);

   DEBUG_SO(fprintf(stderr,"rect_size=0x%04x\n",rect_size);)

   // check if we can skip rect_size
   if(rect_size > end_of_buffer - cursor_normal)
      return IpsOption::NO_MATCH;

   // skip:
   //  rect_size
   //  frame rate  (2 bytes)
   //  frame count (2 bytes)
   cursor_normal += rect_size + 4;

   // loop through up to 50 tags 
   for(unsigned i = 0; i < 50; i++)
   {
      // check if we can read:
      //  tag               (2 bytes)
      //  long form tag_len (4 bytes)
      if(cursor_normal + 6 > end_of_buffer)
         return IpsOption::NO_MATCH;

      tag = read_little_16_inc(cursor_normal);

      tag_type = (tag & 0xFFC0) >> 6;
      tag_len = tag & 0x003F;

      if(tag_len == 0x3F)
      {
         // long form tag_len
         tag_len = read_little_32_inc(cursor_normal);
      }

      DEBUG_SO(fprintf(stderr,"tag_type=0x%04x\n",tag_type);)
      DEBUG_SO(fprintf(stderr,"tag_len=0x%08x\n",tag_len);)

      if(tag_type == 0x56)
      {
         // DefineSceneAndFrameLabelData
         // check if we can read:
         //  scene_count (5 bytes)
         if(cursor_normal + 5 > end_of_buffer)
            return IpsOption::NO_MATCH;

         if((cursor_normal[0] & 0x80) &&
            (cursor_normal[1] & 0x80) &&
            (cursor_normal[2] & 0x80) &&
            (cursor_normal[3] & 0x80) &&
            (cursor_normal[4] & 0x08))
         {
            // vulnerability condition met
            return IpsOption::MATCH;
         }
      }

      // check if we can skip tag_len
      if(tag_len > end_of_buffer - cursor_normal)
         return IpsOption::NO_MATCH;

      // skip tag_len
      cursor_normal += tag_len;
   }

   return IpsOption::NO_MATCH;
}

static SoEvalFunc ctor(const char* /*so*/, void** pv)
{
    *pv = nullptr;
    return eval;
}

static const SoApi so_17647 =
{
    { // base api info
        PT_SO_RULE,
        sizeof(SoApi),
        SOAPI_VERSION,
        8, // version of this file
        API_RESERVED,
        API_OPTIONS,
        "17647", // name
        "FILE-FLASH Adobe Flash Player DefineSceneAndFrameLabelData memory corruption attempt", // help
        nullptr,  // mod_ctor
        nullptr   // mod_dtor
    },
    // so rule api info
    (uint8_t*)rule_17647,
    rule_17647_len,
    nullptr, // pinit
    nullptr, // pterm
    nullptr, // tinit
    nullptr, // tterm
    ctor,    // ctor
    nullptr  // dtor
};

const BaseApi* pso_17647 = &so_17647.base;

