//--------------------------------------------------------------------------
// Copyright (C) 2019-2021 Cisco and/or its affiliates. All rights reserved.
//
// This file may contain proprietary rules that were created, tested and
// certified by Sourcefire, Inc. (the "VRT Certified Rules") as well as
// rules that were created by Sourcefire and other third parties and
// distributed under the GNU General Public License (the "GPL Rules").
// The VRT Certified Rules contained in this file are the property of
// Sourcefire, Inc. Copyright 2005 Sourcefire, Inc. All Rights Reserved.
// The GPL Rules created by Sourcefire, Inc. are the property of
// Sourcefire, Inc. Copyright 2002-2005 Sourcefire, Inc. All Rights
// Reserved. All other GPL Rules are owned and copyrighted by their
// respective owners (please see www.snort.org/contributors for a list
// of owners and their respective copyrights). In order to determine what
// rules are VRT Certified Rules or GPL Rules, please refer to the VRT
// Certified Rules License Agreement.
//--------------------------------------------------------------------------
// file-multimedia_cve-2008-5616-mplayer-demux-open-vqf-bo.cc author Brandon Stultz <brastult@cisco.com>

#include "main/snort_types.h"
#include "framework/so_rule.h"
#include "framework/cursor.h"
#include "protocols/packet.h"
#include "util_read.h"

//#define DEBUG
#ifdef DEBUG
#define DEBUG_SO(code) code
#else
#define DEBUG_SO(code)
#endif

using namespace snort;

static const char* rule_17300 = R"[Snort_SO_Rule](
alert file (
	msg:"FILE-MULTIMEDIA MPlayer demux_open_vqf TwinVQ file handling buffer overflow attempt";
	soid:17300;
	file_data;
	content:"TWIN97012000",depth 12;
	so:eval;
	metadata:policy max-detect-ips drop;
	reference:bugtraq,32822;
	reference:cve,2008-5616;
	classtype:attempted-user;
	gid:3; sid:17300; rev:5;
)
)[Snort_SO_Rule]";

static const unsigned rule_17300_len = 0;

static IpsOption::EvalStatus eval(void*, Cursor& c, Packet* p)
{
   const uint8_t *cursor_normal = c.start(),
                 *end_of_buffer = c.endo(),
                 *end_of_chunk;

   uint32_t chunk_size, chunk_id, tag_len;

   // check if we can read:
   //  chunk_size  (4 bytes)
   //  chunk_id    (4 bytes)
   if(cursor_normal + 8 > end_of_buffer)
      return IpsOption::NO_MATCH;

   chunk_size = read_big_32_inc(cursor_normal);

   // calculate end_of_chunk 
   end_of_chunk = cursor_normal + chunk_size;

   chunk_id = read_big_32_inc(cursor_normal);

   // COMM chunk
   if(chunk_id != 0x434F4D4D)
      return IpsOption::NO_MATCH; 

   // skip:
   //  subchunk_size  (4 bytes)
   //  channel_mode   (4 bytes)
   //  bit_rate       (4 bytes)
   //  BPS            (4 bytes)
   //  security_level (4 bytes)
   cursor_normal += 20;

   // check up to 25 tags
   for(unsigned i = 0; i < 25; i++)
   {
      // skip tag_id (4 bytes)
      cursor_normal += 4;

      // check if we can read:
      //  tag_len (4 bytes)
      if(cursor_normal + 4 > end_of_buffer)
         return IpsOption::NO_MATCH;

      // check if there is data left in the chunk
      if(cursor_normal + 4 > end_of_chunk)
         return IpsOption::NO_MATCH;

      tag_len = read_big_32_inc(cursor_normal);

      // check vulnerability condition
      if(tag_len > chunk_size)
         return IpsOption::MATCH;

      // check if we can skip tag_len
      if(tag_len > end_of_buffer - cursor_normal)
         return IpsOption::NO_MATCH;

      // skip tag_len
      cursor_normal += tag_len;
   }

   return IpsOption::NO_MATCH;
}

static SoEvalFunc ctor(const char* /*so*/, void** pv)
{
    *pv = nullptr;
    return eval;
}

static const SoApi so_17300 =
{
    { // base api info
        PT_SO_RULE,
        sizeof(SoApi),
        SOAPI_VERSION,
        5, // version of this file
        API_RESERVED,
        API_OPTIONS,
        "17300", // name
        "FILE-MULTIMEDIA MPlayer demux_open_vqf TwinVQ file handling buffer overflow attempt", // help
        nullptr,  // mod_ctor
        nullptr   // mod_dtor
    },
    // so rule api info
    (uint8_t*)rule_17300,
    rule_17300_len,
    nullptr, // pinit
    nullptr, // pterm
    nullptr, // tinit
    nullptr, // tterm
    ctor,    // ctor
    nullptr  // dtor
};

const BaseApi* pso_17300 = &so_17300.base;

